/*
 * Decompiled with CFR 0.152.
 */
package com.badlogicgames.packr;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.invoke.MethodHandles;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFilePermission;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Set;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.ArchiveException;
import org.apache.commons.compress.archivers.ArchiveInputStream;
import org.apache.commons.compress.archivers.ArchiveOutputStream;
import org.apache.commons.compress.archivers.ArchiveStreamFactory;
import org.apache.commons.compress.archivers.jar.JarArchiveEntry;
import org.apache.commons.compress.archivers.jar.JarArchiveInputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipFile;
import org.apache.commons.compress.compressors.CompressorException;
import org.apache.commons.compress.compressors.CompressorStreamFactory;
import org.apache.commons.compress.utils.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ArchiveUtils {
    public static final int DEFAULT_FILE_MODE = 420;
    public static final int DEFAULT_DIRECTORY_MODE = 493;
    public static final int DEFAULT_LINK_MODE = 511;
    public static final int ZIP_LINK_FLAG = 40960;
    public static final int OWNER_READ_BIT_MASK = 256;
    public static final int OWNER_WRITE_BIT_MASK = 128;
    public static final int OWNER_EXECUTE_BIT_MASK = 64;
    public static final int GROUP_READ_BIT_MASK = 32;
    public static final int GROUP_WRITE_BIT_MASK = 16;
    public static final int GROUP_EXECUTE_BIT_MASK = 8;
    public static final int OTHERS_READ_BIT_MASK = 4;
    public static final int OTHERS_WRITE_BIT_MASK = 2;
    public static final int OTHERS_EXECUTE_BIT_MASK = 1;
    private static final Logger LOG = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());

    private ArchiveUtils() {
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static void extractArchive(Path archivePath, Path extractToDirectory) throws IOException, CompressorException, ArchiveException {
        try (BufferedInputStream jdkInputStream = new BufferedInputStream(Files.newInputStream(archivePath, new OpenOption[0]));){
            String compressorType = null;
            try {
                compressorType = CompressorStreamFactory.detect((InputStream)jdkInputStream);
            }
            catch (CompressorException exception) {
                LOG.debug("Didn't detect any compression for archive " + archivePath + ": " + exception.getMessage());
            }
            BufferedInputStream decompressedJdkInputStream = jdkInputStream;
            if (compressorType != null) {
                decompressedJdkInputStream = new BufferedInputStream((InputStream)CompressorStreamFactory.getSingleton().createCompressorInputStream(compressorType, (InputStream)jdkInputStream));
            }
            switch (ArchiveStreamFactory.detect((InputStream)decompressedJdkInputStream)) {
                case "zip": {
                    if (compressorType != null) {
                        LOG.error("Cannot extract Zip archives that are wrapped in additional compression");
                        return;
                    } else {
                        ArchiveUtils.extractZipArchive(archivePath, extractToDirectory);
                        return;
                    }
                }
                case "jar": {
                    ArchiveUtils.extractJarArchive(decompressedJdkInputStream, extractToDirectory);
                    return;
                }
                case "tar": {
                    ArchiveUtils.extractTarArchive(decompressedJdkInputStream, extractToDirectory);
                    return;
                }
                default: {
                    LOG.error("No special handling for archive type " + archivePath + ". Permissions and links will not be properly handled.");
                    ArchiveUtils.extractGenericArchive(decompressedJdkInputStream, extractToDirectory);
                    return;
                }
            }
        }
    }

    private static void extractGenericArchive(InputStream inputStream, Path extractToDirectory) throws ArchiveException, IOException {
        ArchiveEntry entry;
        ArchiveInputStream archiveInputStream = new ArchiveStreamFactory().createArchiveInputStream(inputStream);
        while ((entry = archiveInputStream.getNextEntry()) != null) {
            if (!archiveInputStream.canReadEntryData(entry)) {
                LOG.error("Failed to read archive entry " + entry);
                continue;
            }
            Path entryExtractPath = extractToDirectory.resolve(ArchiveUtils.getEntryAsPath(entry));
            if (entry.isDirectory()) {
                Files.createDirectories(entryExtractPath, new FileAttribute[0]);
            } else {
                Files.createDirectories(entryExtractPath.getParent(), new FileAttribute[0]);
                Files.copy((InputStream)archiveInputStream, entryExtractPath, StandardCopyOption.REPLACE_EXISTING);
            }
            Files.setLastModifiedTime(entryExtractPath, FileTime.fromMillis(entry.getLastModifiedDate().getTime()));
        }
    }

    private static void extractTarArchive(InputStream inputStream, Path extractToDirectory) throws IOException {
        TarArchiveEntry entry;
        TarArchiveInputStream archiveInputStream = new TarArchiveInputStream(inputStream);
        while ((entry = archiveInputStream.getNextTarEntry()) != null) {
            Path linkTarget;
            if (!archiveInputStream.canReadEntryData((ArchiveEntry)entry)) {
                LOG.error("Failed to read archive entry " + entry);
                continue;
            }
            Path entryExtractPath = extractToDirectory.resolve(ArchiveUtils.getEntryAsPath((ArchiveEntry)entry));
            if (entry.isLink()) {
                linkTarget = Paths.get(entry.getLinkName(), new String[0]);
                Files.deleteIfExists(entryExtractPath);
                Files.createLink(entryExtractPath, linkTarget);
            } else if (entry.isSymbolicLink()) {
                linkTarget = Paths.get(entry.getLinkName(), new String[0]);
                Files.deleteIfExists(entryExtractPath);
                Files.createSymbolicLink(entryExtractPath, linkTarget, new FileAttribute[0]);
            } else if (entry.isDirectory()) {
                Files.createDirectories(entryExtractPath, new FileAttribute[0]);
            } else {
                Files.createDirectories(entryExtractPath.getParent(), new FileAttribute[0]);
                Files.copy((InputStream)archiveInputStream, entryExtractPath, StandardCopyOption.REPLACE_EXISTING);
            }
            ArchiveUtils.setLastModifiedTime(entryExtractPath, FileTime.fromMillis(entry.getLastModifiedDate().getTime()));
            Set<PosixFilePermission> permissions = ArchiveUtils.getPosixFilePermissions(entry);
            ArchiveUtils.setPosixPermissions(entryExtractPath, permissions);
        }
    }

    private static Set<PosixFilePermission> getPosixFilePermissions(TarArchiveEntry entry) {
        int mode = entry.getMode();
        if (mode == 0) {
            mode = entry.isSymbolicLink() ? 511 : (entry.isDirectory() ? 493 : 420);
        }
        return ArchiveUtils.getPosixFilePermissions(mode);
    }

    private static Set<PosixFilePermission> getPosixFilePermissions(ZipArchiveEntry entry) {
        int mode = entry.getUnixMode();
        if (mode == 0) {
            mode = entry.isUnixSymlink() ? 511 : (entry.isDirectory() ? 493 : 420);
        }
        return ArchiveUtils.getPosixFilePermissions(mode);
    }

    private static Set<PosixFilePermission> getPosixFilePermissions(int mode) {
        HashSet<PosixFilePermission> permissions = new HashSet<PosixFilePermission>();
        if ((mode & 0x100) != 0) {
            permissions.add(PosixFilePermission.OWNER_READ);
        }
        if ((mode & 0x80) != 0) {
            permissions.add(PosixFilePermission.OWNER_WRITE);
        }
        if ((mode & 0x40) != 0) {
            permissions.add(PosixFilePermission.OWNER_EXECUTE);
        }
        if ((mode & 0x20) != 0) {
            permissions.add(PosixFilePermission.GROUP_READ);
        }
        if ((mode & 0x10) != 0) {
            permissions.add(PosixFilePermission.GROUP_WRITE);
        }
        if ((mode & 8) != 0) {
            permissions.add(PosixFilePermission.GROUP_EXECUTE);
        }
        if ((mode & 4) != 0) {
            permissions.add(PosixFilePermission.OTHERS_READ);
        }
        if ((mode & 2) != 0) {
            permissions.add(PosixFilePermission.OTHERS_WRITE);
        }
        if ((mode & 1) != 0) {
            permissions.add(PosixFilePermission.OTHERS_EXECUTE);
        }
        return permissions;
    }

    private static void setPosixPermissions(Path path, Set<PosixFilePermission> permissions) throws IOException {
        if (Files.isSymbolicLink(path)) {
            return;
        }
        PosixFileAttributeView posixFileAttributeView = Files.getFileAttributeView(path, PosixFileAttributeView.class, LinkOption.NOFOLLOW_LINKS);
        if (posixFileAttributeView != null) {
            posixFileAttributeView.setPermissions(permissions);
        }
    }

    private static void extractJarArchive(InputStream inputStream, Path extractToDirectory) throws IOException {
        JarArchiveEntry entry;
        JarArchiveInputStream archiveInputStream = new JarArchiveInputStream(inputStream);
        while ((entry = archiveInputStream.getNextJarEntry()) != null) {
            if (!archiveInputStream.canReadEntryData((ArchiveEntry)entry)) {
                LOG.error("Failed to read archive entry " + entry);
                continue;
            }
            ArchiveUtils.extractZipEntry(extractToDirectory, (InputStream)archiveInputStream, (ZipArchiveEntry)entry);
        }
    }

    private static void extractZipEntry(Path extractToDirectory, InputStream archiveInputStream, ZipArchiveEntry entry) throws IOException {
        Path entryExtractPath = extractToDirectory.resolve(ArchiveUtils.getEntryAsPath((ArchiveEntry)entry));
        if (entry.isUnixSymlink()) {
            byte[] contentBuffer = new byte[8192];
            int contentLength = IOUtils.readFully((InputStream)archiveInputStream, (byte[])contentBuffer);
            Path linkTarget = Paths.get(new String(contentBuffer, 0, contentLength, StandardCharsets.UTF_8), new String[0]);
            Files.deleteIfExists(entryExtractPath);
            Files.createSymbolicLink(entryExtractPath, linkTarget, new FileAttribute[0]);
        } else if (entry.isDirectory()) {
            Files.createDirectories(entryExtractPath, new FileAttribute[0]);
        } else {
            Files.createDirectories(entryExtractPath.getParent(), new FileAttribute[0]);
            Files.copy(archiveInputStream, entryExtractPath, StandardCopyOption.REPLACE_EXISTING);
        }
        ArchiveUtils.setLastModifiedTime(entryExtractPath, entry.getLastModifiedTime());
        Set<PosixFilePermission> permissions = ArchiveUtils.getPosixFilePermissions(entry);
        ArchiveUtils.setPosixPermissions(entryExtractPath, permissions);
    }

    private static void setLastModifiedTime(Path path, FileTime lastModifiedTime) throws IOException {
        if (Files.isSymbolicLink(path)) {
            return;
        }
        BasicFileAttributeView pathAttributeView = Files.getFileAttributeView(path, BasicFileAttributeView.class, LinkOption.NOFOLLOW_LINKS);
        BasicFileAttributes fileAttributes = pathAttributeView.readAttributes();
        pathAttributeView.setTimes(lastModifiedTime, fileAttributes.lastAccessTime(), fileAttributes.creationTime());
    }

    private static Path getEntryAsPath(ArchiveEntry entry) throws IOException {
        Path entryAsPath = Paths.get(entry.getName(), new String[0]);
        if (entryAsPath.isAbsolute()) {
            throw new IOException("Archive contained an absolute path as an entry");
        }
        return entryAsPath;
    }

    private static void extractZipArchive(Path archivePath, Path extractToDirectory) throws IOException {
        try (ZipFile zipFile = new ZipFile(archivePath.toFile());){
            Enumeration entries = zipFile.getEntries();
            while (entries.hasMoreElements()) {
                ZipArchiveEntry entry = (ZipArchiveEntry)entries.nextElement();
                InputStream entryInputStream = zipFile.getInputStream(entry);
                try {
                    ArchiveUtils.extractZipEntry(extractToDirectory, entryInputStream, entry);
                }
                finally {
                    if (entryInputStream == null) continue;
                    entryInputStream.close();
                }
            }
        }
    }

    public static void createArchive(final ArchiveType archiveType, final Path directoryToArchive, Path archiveFile) throws IOException, ArchiveException {
        try (BufferedOutputStream fileOutputStream = new BufferedOutputStream(Files.newOutputStream(archiveFile, new OpenOption[0]));
             final ArchiveOutputStream archiveOutputStream = new ArchiveStreamFactory().createArchiveOutputStream(archiveType.getCommonsCompressName(), (OutputStream)fileOutputStream);){
            if (archiveType == ArchiveType.TAR) {
                ((TarArchiveOutputStream)archiveOutputStream).setLongFileMode(2);
            }
            Files.walkFileTree(directoryToArchive, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    ArchiveUtils.createAndPutArchiveEntry(archiveType, archiveOutputStream, directoryToArchive, file);
                    archiveOutputStream.closeArchiveEntry();
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    if (Files.isSameFile(dir, directoryToArchive)) {
                        return FileVisitResult.CONTINUE;
                    }
                    ArchiveEntry entry = archiveOutputStream.createArchiveEntry(dir.toFile(), ArchiveUtils.getRelativePathString(dir, directoryToArchive));
                    archiveOutputStream.putArchiveEntry(entry);
                    archiveOutputStream.closeArchiveEntry();
                    return FileVisitResult.CONTINUE;
                }
            });
            archiveOutputStream.finish();
        }
    }

    private static void createAndPutArchiveEntry(ArchiveType archiveType, ArchiveOutputStream archiveOutputStream, Path directoryToArchive, Path filePathToArchive) throws IOException {
        switch (archiveType) {
            case ZIP: {
                ZipArchiveEntry entry = new ZipArchiveEntry(filePathToArchive.toFile(), ArchiveUtils.getRelativePathString(filePathToArchive, directoryToArchive));
                entry.setUnixMode(ArchiveUtils.getUnixMode(filePathToArchive));
                boolean isSymbolicLink = Files.isSymbolicLink(filePathToArchive);
                if (isSymbolicLink) {
                    entry.setUnixMode(entry.getUnixMode() | 0xA000);
                }
                archiveOutputStream.putArchiveEntry((ArchiveEntry)entry);
                if (isSymbolicLink) {
                    archiveOutputStream.write(ArchiveUtils.getRelativePathString(Files.readSymbolicLink(filePathToArchive), directoryToArchive).getBytes(StandardCharsets.UTF_8));
                    break;
                }
                Files.copy(filePathToArchive, (OutputStream)archiveOutputStream);
                break;
            }
            case TAR: {
                TarArchiveEntry entry;
                boolean isSymbolicLink = Files.isSymbolicLink(filePathToArchive);
                if (isSymbolicLink) {
                    entry = new TarArchiveEntry(ArchiveUtils.getRelativePathString(filePathToArchive, directoryToArchive), 50);
                    entry.setLinkName(ArchiveUtils.getRelativePathString(Files.readSymbolicLink(filePathToArchive), directoryToArchive));
                } else {
                    entry = new TarArchiveEntry(filePathToArchive.toFile(), ArchiveUtils.getRelativePathString(filePathToArchive, directoryToArchive));
                }
                entry.setMode(ArchiveUtils.getUnixMode(filePathToArchive));
                archiveOutputStream.putArchiveEntry((ArchiveEntry)entry);
                if (isSymbolicLink) break;
                Files.copy(filePathToArchive, (OutputStream)archiveOutputStream);
                break;
            }
        }
    }

    private static int getUnixMode(Path file) throws IOException {
        PosixFileAttributeView fileAttributeView = Files.getFileAttributeView(file, PosixFileAttributeView.class, LinkOption.NOFOLLOW_LINKS);
        if (fileAttributeView == null) {
            if (Files.isSymbolicLink(file)) {
                return 511;
            }
            if (Files.isDirectory(file, new LinkOption[0])) {
                return 493;
            }
            return 420;
        }
        int mode = 0;
        Set<PosixFilePermission> permissions = fileAttributeView.readAttributes().permissions();
        if (permissions.contains((Object)PosixFilePermission.OWNER_READ)) {
            mode |= 0x100;
        }
        if (permissions.contains((Object)PosixFilePermission.OWNER_WRITE)) {
            mode |= 0x80;
        }
        if (permissions.contains((Object)PosixFilePermission.OWNER_EXECUTE)) {
            mode |= 0x40;
        }
        if (permissions.contains((Object)PosixFilePermission.GROUP_READ)) {
            mode |= 0x20;
        }
        if (permissions.contains((Object)PosixFilePermission.GROUP_WRITE)) {
            mode |= 0x10;
        }
        if (permissions.contains((Object)PosixFilePermission.GROUP_EXECUTE)) {
            mode |= 8;
        }
        if (permissions.contains((Object)PosixFilePermission.OTHERS_READ)) {
            mode |= 4;
        }
        if (permissions.contains((Object)PosixFilePermission.OTHERS_WRITE)) {
            mode |= 2;
        }
        if (permissions.contains((Object)PosixFilePermission.OTHERS_EXECUTE)) {
            mode |= 1;
        }
        LOG.trace("Unix mode of file=" + file + ", mode=" + Integer.toOctalString(mode) + ", permissions=" + permissions);
        return mode;
    }

    private static String getRelativePathString(Path path, Path rootDirectory) {
        String relativePathString = rootDirectory.relativize(path).toString().replaceAll("\\\\", "/");
        LOG.trace("Creating relative path from path=" + path + ", rootDirectory=" + rootDirectory + ", relativePathString=" + relativePathString);
        return relativePathString;
    }

    public static enum ArchiveType {
        ZIP("zip"),
        TAR("tar");

        private final String commonsCompressName;

        private ArchiveType(String commonsCompressName) {
            this.commonsCompressName = commonsCompressName;
        }

        public String getCommonsCompressName() {
            return this.commonsCompressName;
        }
    }
}

